classdef SurfaceMesh < handle
    % Mesh of triangles for surfaces
    %   
    % Properties
    %
    %   Vertices   = vct3Array(0) % Vertices - vct3Vector of vertex points
    %   Triangles  = []     Triangles -stored as a 2D array of vertex indices
    %                           [[v,v,v];[v,v,v];...;[v,v,v]]
    %   Neighbors  = []     Neighbors - stored as 2D array of triange indices
    %                           [[n,n,n];[n,n,n];...;[n,n,n]]
    %   Meshfile   = ''     filename for mesh if mesh was read in
    %
    % Methods
    %
    % M = SurfaceMesh()         Creates empty mesh
    % M = SurfaceMesh('fid')    Read mesh from file 'fid' (see ReadSurFile)
    % M = SurfaceMesh(V,T,N)    Create from known elements
    %                           V = Vct3Array of vertices
    %                           T = Array of vertex indices 
    %                             = [... ; [p,q,r]; ... ]
    %                           N = Array of neighbor triangle indices
    %                             = [... ; [np,nq,nr] ; ... ]
    % [TV,p,q,t] = M.Triangle(t) Returns elements of triangle t
    %                           vct3Array TV    = [p,q,r]
    %                           vct3 p, q, r    = vertices 
    % [cp,ct,cd] = M.LinearSearchForClosestPoint(c)
    %                           Serch M for closest point to c
    %                           vct3    cp = closest point
    %                           int     ct = triangle number
    %                           double  cd = distance from c to cp
    % [v,t] = M.Sample()    Randomly sample point on mesh
    %                           vct3    v = point on triangle
    %                           int     t = triangle index containing v
    % [v,t] = M.Sample(BB)  Randomly sample point on mesh and add random
    %                       displacement from vct3BoundingBox B 
    % [v,t] = M.Sample(A)   Randomly sample point on mesh and add random
    %                       displacement vct3.rand(-A,A)
    % [v,t] = M.Sample(A,B) Randomly sample point on mesh and add random
    %                       displacement vct3.rand(A,B)
    %                       Note: A & B muct be vct3 in this case
    % [V,T] = M.Sample(A,N) Produces vct3Array of sample points 
    %                       of the form M.Sample(A).  
    %                           vct3Array V     = Sample points
    %                           int vector T    = triangle indices
    % [V,T] = M.Sample(A,B,N) Produces vct3Array of sample points 
    %                         of the form M.Sample(A,B).  
    %                           vct3Array V     = Sample points
    %                           int vector T    = triangle indices
    % M = SurfaceMesh.ReadSurFile('fid') reads surface mesh file.  This
    %                       is a text file with the following format
    %                           Nvertices       - Number of vertices
    %                           x1,y1,z1        - vertex coordinates
    %                              ...
    %                           xN,yN,zN        - vertex coordinates
    %                           Ntriangles      - num triangles
    %                           p1,q1,r1,np1,nq1,nr1 - vertex and nbr indcs
    %                              ...
    %                           pN,qN,rN,npN,nqN,nrN - vertex and nbr indcs
    %                       NOTE: The indexing in this file is 0 base, 
    %                           even though MATLAB uses 1 base indexing.
    %                           The read utility automatically makes the
    %                           conversion.
    %
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor

    
    properties
        Vertices   = vct3Array(0) % Vertices - vct3Vector of vertex points
        Triangles  = []     % Triangles -stored as a 2D array of vertex indices
                            %     [[v,v,v];[v,v,v];...;[v,v,v]]
        Neighbors  = []     % Neighbors - stored as 2D array of triange indices
                            %     [[n,n,n];[n,n,n];...;[n,n,n]]
        Meshfile   = ''     % filename for mesh if mesh was read in
    end
    
    methods
        
        function M = SurfaceMesh(V,T,N)
            if nargin==0
                return;
            end;
            if isa(V,'SurfaceMesh')
                % copy constructor
                M.Vertices = V.Vertices;
                M.Triangles = V.Triangles
                M.Neighbors = V.Neighbors
            else
                M.Vertices = V;
                M.Triangles = T;
                
                if nargin > 2
                    M.Neighbors = N;
                end
            end
        end
        
        function [TV,p,q,r] = Triangle(S,t)
            T = S.Triangles(t,:);
            p = S.Vertices(T(1));
            q = S.Vertices(T(2));
            r = S.Vertices(T(3));
            TV = vct3Array([p.el,q.el,r.el]);
        end
        
        function [cp,ct,cd] = LinearSearchForClosestPoint(S,c)
            cd = inf; ct = 0;
            for t=1:size(S.Triangles,1)
                T = S.Triangles(t,:);
                [p,d] = TriangleClosestPoint(c, ...
                                             S.Vertices(T(1)), ...
                                             S.Vertices(T(2)), ...
                                             S.Vertices(T(3)));
                if d<cd
                    cd = d; cp = p; ct=t;
                    if d==0 
                        return;
                    end
                end
            end
        end  
        
        function [v,t] = Sample(S,A,B,N)
            switch nargin
                case 1
                    Dmin=vct3;
                    Dmax=vct3;
                case 2
                    if isa(A,'vct3')
                        Dmax = A; Dmin = -A;
                    elseif isa(A,'vct3BoundingBox')
                        Dmax = A.vMax; Dmin = A.vMin;
                    elseif isscalar(A)
                        d=abs(A);Dmax = vct3(d,d,d); Dmin = -Dmax;
                    else
                        error('BadArg','Illegal argument to Mesh Sample');
                    end
                case 3
                    if isa(B,'vct3')
                        % assume A and B are noise limits
                        Dmin = B; Dmax = B;
                    else
                        % B is a scalar, assume it is a count
                        if isa(A,'vct3')
                            Dmax = A
                        else
                            d = abs(A); 
                            Dmax = vct3(d,d,d);
                        end
                        Dmin = - Dmax;
                        [v,t] = Sample(S,Dmin,Dmax);
                        return;
                    end
                case 4
                    v = vct3Array(N);
                    t = zeros(1,N);
                    if isa(A,'vct3')
                        Dmin = A;
                    else 
                        Dmin = vct3(A,A,A);
                    end
                    if isa(B,'vct3')
                        Dmax = B;
                    else
                        Dmax = vct3(B,B,B);
                    end
                        
                    for i=1:N
                        [v(i),t(i)] = S.Sample(Dmin,Dmax);
                    end 
                    return;
                otherwise
                    error('BadArg','Illegal argument to Mesh Sample');
            end;
            Nt = size(S.Triangles,1);
            t = min(Nt,max(int32(Nt*rand()),1)); 
            TV = S.Triangle(t);
            Lam = rand(3,1);
            Lam = Lam/norm(Lam,1);
            v = vct3(TV.el*Lam);D = Dmin.el+rand(3,1).*(Dmax.el-Dmin.el);
            [U,S,V]=svd(TV*TV');
            Dv = vct3(V*U'*D);
            v = v + Dv;         
        end
    end
    
    methods (Static)
        function M = ReadSurFile(fid)
            f = fopen(fid,'r');
            M = SurfaceMesh;
            M.Meshfile = fid;
            Nv = fscanf(f,'%d',1);
            M.Vertices.el = fscanf(f,'%f',[3,Nv]);
            Nt = fscanf(f,'%d',1);
            TN = fscanf(f,'%d',[6,Nt]);
            M.Triangles = TN(1:3,1:Nt)'+ones(Nt,3);
            M.Neighbors = TN(4:6,1:Nt)'+ones(Nt,3);
            fclose(f);                  
        end
    end
    
end

